/*********************************************************************
*	 Project: MG82F6P32-DEMO
*			TH253A MG82F6P32_LQFP32_QFN32_EV_V11 EV	Board
*			CpuCLK=12MHz, SysCLK=12MHz
*	 Description:
*			Through	the	IVREF 1.4v,	calculate the P10 input	voltage
*	 Note:
*
*	 Creat time:2024.02.26
*	 Modify:
*
*********************************************************************/

#define	_MAIN_C

#include <Intrins.h>
#include <Absacc.h>

#include <Stdio.h>	// for printf

#include ".\include\REG_MG82F6P32.H"
#include ".\include\Type.h"
#include ".\include\API_Macro_MG82F6P32.H"
#include ".\include\API_Uart_BRGRL_MG82F6P32.H"

/*************************************************
*Set SysClk	(MAX.50MHz)
*Selection:
*	11059200,12000000,
*	22118400,24000000,
*	29491200,32000000,
*	44236800,48000000
*************************************************/
#define	MCU_SYSCLK		12000000
/*************************************************/
/*************************************************
*Set CpuClk	(MAX.36MHz)
*	1) CpuCLK=SysCLK
*	2) CpuClk=SysClk/2
*************************************************/

#define	MCU_CPUCLK		(MCU_SYSCLK)
//#define MCU_CPUCLK		(MCU_SYSCLK/2)

#define	TIMER_1T_1ms_TH	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) /256)
#define	TIMER_1T_1ms_TL	((65536-(u16)(float)(1000*((float)(MCU_SYSCLK)/(float)(1000000)))) %256)

#define	IO_LED_R		P22
#define	IO_LED_Y		P24
#define	IO_LED_G		P26

#define	VCC_REF					(3300L)

/*************************************************
*Function:		char putchar (char c)
*Description:	send for printf
*Input:			char c
*Output:
*************************************************/
char putchar (char c)
{
	bit	bES;
	bES=ES0;
	ES0=0;
	S0BUF =	c;
	while(TI0==0);
	TI0=0;
	ES0=bES;
	return 0;
}

/*************************************************
*Function:		void DelayXus(u16 xUs)
*Description:		delay,unit:us
*Input:				u8 Us -> *1us  (1~255)
*Output:
*************************************************/
void DelayXus(u8 xUs)
{
	while(xUs!=0)
	{
#if	(MCU_CPUCLK>=11059200)
		_nop_();
#endif
#if	(MCU_CPUCLK>=14745600)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=16000000)
		_nop_();
#endif

#if	(MCU_CPUCLK>=22118400)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=24000000)
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=29491200)
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
		_nop_();
#endif
#if	(MCU_CPUCLK>=32000000)
		_nop_();
		_nop_();
#endif

		xUs--;
	}
}

/*************************************************
*Function:		void DelayXms(u16 xMs)
*Description:	 delay,unit:ms
*Input:				u16	xMs	-> *1ms	 (1~65535)
*Output:
*************************************************/
void DelayXms(u16 xMs)
{
	while(xMs!=0)
	{
		CLRWDT();
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		DelayXus(200);
		xMs--;

	}
}

/***********************************************************************************
*Function:		void InitUart0_S0BRG()
*Description:	Initialize Uart0, The source of	baud rate was S0BRG
*Input:
*Output:
*************************************************************************************/
void InitUart0_S0BRG(void)
{
	UART0_SetMode8bitUARTVar();			// UART0 Mode: 8-bit, Variable B.R.
	UART0_EnReception();				// Enable reception
	UART0_SetBRGFromS0BRG();			// B.R.	source:	S0BRG
	UART0_SetS0BRGBaudRateX2();			// S0BRG x2
	UART0_SetS0BRGSelSYSCLK();			// S0BRG clock source: SYSCLK
	UART0_SetRxTxP30P31();
	// Sets	B.R. value
	UART0_SetS0BRGValue(S0BRG_BRGRL_9600_2X_12000000_1T);

	UART0_EnS0BRG();					// Enable S0BRG
}

/***********************************************************************************
*Function:		void InitPort()
*Description:	Initialize IO Port
*Input:
*Output:
*************************************************************************************/
void InitPort(void)
{
	PORT_SetP2PushPull(BIT2|BIT4|BIT6);				// Set P22,P24,P26 as push-pull(LED)
	PORT_SetP3QuasiBi(BIT0|BIT1);					// Set P30,P31 as Quasi-Bi
	PORT_SetP1AInputOnly(BIT0);				// Set P10(AIN0) as	Analog-Input-Only for ADC input
}

/***********************************************************************************
*Function:		void InitADC(void)
*Description:		Initialize ADC
*Input:
*Output:
*************************************************************************************/
void InitADC(void)
{
	ADC_Enable();									// Enable ADC
	ADC_SetADCConversion_30ADCClock();
	ADC_SetClock_SYSCLKDiv8();						// ADC Clock = SYSCLK/8		  sps= 12M/8/30=50K
	ADC_SetMode_SetADCS();							// ADC Trigger mode: set ADCS
	ADC_SetADCData_12Bit();							// ADC ADC Data	resolution:	12bit
	ADC_SetRightJustified();						// ADC Right-Justified
	ADC_SetVREF_IVR24();							// ADC VREF+ is	IVR	2.4V
	ADC_SetInput_AIN();								// ADC input is	AIN
	IVREF_Enable();									// Enable IVR 2.4V
}

/***********************************************************************************
*Function:		u16	GetAdcValue()
*Description:	read ADC result	of the current channel
*Input:
*Output:		u16	:ADC Value
*************************************************************************************/
u16	GetAdcValue()
{
	WordTypeDef	wAdcValue;
	ADCON0 = ADCON0|ADCS;				// set ADCS,Start ADC
	while((ADCON0&ADCI)==0);			// wait	ADC	complete
	wAdcValue.B.BHigh=ADCDH;
	wAdcValue.B.BLow=ADCDL;
	ADCON0 = ADCON0&(~ADCI);			//clear	ADCI flag
	return wAdcValue.W&0x0FFF;
}

/***********************************************************************************
*Function:		u16	GetIVRef()
*Description:	Read the calibration value of the IVREF
*Input:
*Output:		u16	:ADC Value
*************************************************************************************/
u16	GetIVRef()
{
	WordTypeDef	IVR_Code;
	ISPCR =	0x80;
	IFMT = 0x06;
	IFADRH = 0x00;
	IFADRL = 0xC0;

	SCMD = 0x46;
	SCMD = 0xB9;
	IVR_Code.B.BHigh = IFD;

	IFADRL ++;
	SCMD = 0x46;
	SCMD = 0xB9;
	IVR_Code.B.BLow= IFD;

	ISPCR =	0x00;

	return IVR_Code.W;
}

/***********************************************************************************
*Function:		void InitClock()
*Description:	Initialize clock
*Input:
*Output:
*************************************************************************************/
void InitClock(void)
{
#if	(MCU_SYSCLK==11059200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=11.0592MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=11.0592MHz CpuClk=5.5296MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==12000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=12MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1);

#else
	// SysClk=12MHz	CpuClk=6MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1);
#endif
#endif

#if	(MCU_SYSCLK==22118400)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=22.1184MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=22.1184MHz CpuClk=11.0592MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==24000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// SysClk=24MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#else
	// SysClk=24MHz	CpuClk=12MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx4, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X4|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==29491200)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=29.491200MHz CpuClk=29.491200MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=29.491200MHz CpuClk=14.7456MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==32000000)
#if	(MCU_CPUCLK==MCU_SYSCLK)
	// Cpuclk high speed
	CLK_SetCpuCLK_HighSpeed();
	// SysClk=32MHz	CpuClk=32MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_1|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#else
	// SysClk=32MHz	CpuClk=16MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx5.33, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X533|OSCIn_IHRCO);
#endif
#endif

#if	(MCU_SYSCLK==44236800)
	// SysClk=44.2368MHz CpuClk=22.1184MHz
	CLK_SetCKCON0(IHRCO_110592MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

#if	(MCU_SYSCLK==48000000)
	// SysClk=48MHz	CpuClk=24MHz
	CLK_SetCKCON0(IHRCO_12MHz|CPUCLK_SYSCLK_DIV_2|SYSCLK_MCKDO_DIV_1|ENABLE_CKM|CKM_OSCIN_DIV_2);
	DelayXus(100);
	// IHRCO, MCK=CKMIx8, OSCin=IHRCO
	CLK_SetCKCON2(ENABLE_IHRCO|MCK_CKMI_X8|OSCIn_IHRCO);
#endif

	// P60 output MCK/4
	//CLK_P60OC_MCKDiv4();
}

/***********************************************************************************
*Function:		 void InitSystem(void)
*Description:	 Initialize	MCU
*Input:
*Output:
*************************************************************************************/
void InitSystem(void)
{
	InitPort();
	InitClock();
	InitUart0_S0BRG();
	InitADC();
}

void main()
{
	u8 i;
	DWordTypeDef dwTemp;
	WordTypeDef	wVCCValue,wVinValue,wIVRefValue;
	WordTypeDef	wIVRAdjAdc,wVDDAdc,wVinAdc;

	InitSystem();

	IO_LED_G=0;IO_LED_R=0,IO_LED_Y=0;
	DelayXms(1000);
	IO_LED_G=1;IO_LED_R=1,IO_LED_Y=1;

	wIVRAdjAdc.W=GetIVRef();

	// Check the calibration value is within the range(2.3~2.5)
	if((wIVRAdjAdc.W <2854)||(wIVRAdjAdc.W>=3103))
	{ // There is something	wrong with the stored value.Set	to the default value 2.4V@3.3V
		// Uart	output
		printf("\nErr:%d",wIVRAdjAdc.W);
		IO_LED_R=0;
		wIVRAdjAdc.W=2979;
	}
	else
	{
		// Uart	output
		printf("\nIVR ADC@3.3V:%d",wIVRAdjAdc.W);
	}
	// Calculate the IVR voltage
	dwTemp.DW=wIVRAdjAdc.W;
	wIVRefValue.W=(dwTemp.DW*VCC_REF)/4096;
	printf("\nIVRef:%d",wIVRefValue.W);


	while(1)
	{
		IO_LED_G=0;
		DelayXms(200);
		IO_LED_G=1;
		DelayXms(200);

		ADC_SetChannel_VDD_Div4();	// set ADC channel is VDD/4
		// Read	ADC	values 16 times	and	calculate the average value
		wVDDAdc.W=0x0000;
		for(i=0;i<16;i++)
		{
			wVDDAdc.W=wVDDAdc.W+GetAdcValue();
		}
		wVDDAdc.W=wVDDAdc.W>>4;

		// Calculate the current VCC value
		dwTemp.DW=wIVRefValue.W;
		wVCCValue.W=(dwTemp.DW*wVDDAdc.W)/(4096L/4);

		// Uart	output
		printf("\nVCC:%d",wVCCValue.W);

		ADC_SetChannel_AIN0();		// Set the ADC channel to AIN0
		// calculate the P10(AIN0) input voltage
		wVinAdc.W=0x0000;
		for(i=0;i<16;i++)
		{
			wVinAdc.W=wVinAdc.W+GetAdcValue();
		}
		wVinAdc.W=wVinAdc.W>>4;

		// Calculate the input voltage of AIN0
		dwTemp.DW=wIVRefValue.W;
		wVinValue.W=dwTemp.DW*wVinAdc.W/(4096L);

		// Output the result by	UART
		printf(" Vin:%d",wVinValue.W);
	}
}


